/*
 * Decompiled with CFR 0.152.
 */
package com.metamx.tranquility.kafka;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.metamx.common.logger.Logger;
import com.metamx.tranquility.config.DataSourceConfig;
import com.metamx.tranquility.kafka.model.MessageCounters;
import com.metamx.tranquility.kafka.model.PropertiesBasedKafkaConfig;
import com.metamx.tranquility.kafka.writer.WriterController;
import io.druid.concurrent.Execs;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import kafka.consumer.Consumer;
import kafka.consumer.ConsumerConfig;
import kafka.consumer.ConsumerIterator;
import kafka.consumer.KafkaStream;
import kafka.consumer.TopicFilter;
import kafka.consumer.Whitelist;
import kafka.javaapi.consumer.ConsumerConnector;
import kafka.message.MessageAndMetadata;

public class KafkaConsumer {
    private static final Logger log = new Logger(KafkaConsumer.class);
    private final ExecutorService consumerExec;
    private final Thread commitThread;
    private final AtomicBoolean shutdown = new AtomicBoolean();
    private final ReentrantReadWriteLock commitLock = new ReentrantReadWriteLock();
    private final ConsumerConnector consumerConnector;
    private final TopicFilter topicFilter;
    private final int numThreads;
    private final int commitMillis;
    private final WriterController writerController;
    private Map<String, MessageCounters> previousMessageCounters = new HashMap<String, MessageCounters>();

    public KafkaConsumer(PropertiesBasedKafkaConfig propertiesBasedKafkaConfig, Properties properties, Map<String, DataSourceConfig<PropertiesBasedKafkaConfig>> map, WriterController writerController) {
        this.consumerConnector = KafkaConsumer.getConsumerConnector(properties);
        this.topicFilter = new Whitelist(KafkaConsumer.buildTopicFilter(map));
        log.info("Kafka topic filter [%s]", new Object[]{this.topicFilter});
        int n = Math.max(1, Runtime.getRuntime().availableProcessors() - 1);
        this.numThreads = propertiesBasedKafkaConfig.getConsumerNumThreads() > 0 ? propertiesBasedKafkaConfig.getConsumerNumThreads() : n;
        this.commitMillis = propertiesBasedKafkaConfig.getCommitPeriodMillis();
        this.writerController = writerController;
        this.consumerExec = Execs.multiThreaded((int)this.numThreads, (String)"KafkaConsumer-%d");
        this.commitThread = new Thread(this.createCommitRunnable());
        this.commitThread.setName("KafkaConsumer-CommitThread");
        this.commitThread.setDaemon(true);
    }

    public void start() {
        this.commitThread.start();
        this.startConsumers();
    }

    public void stop() {
        if (this.shutdown.compareAndSet(false, true)) {
            log.info("Shutting down - attempting to flush buffers and commit final offsets", new Object[0]);
            try {
                this.commitLock.writeLock().lockInterruptibly();
                try {
                    this.writerController.flushAll();
                    this.writerController.stop();
                    this.consumerConnector.commitOffsets();
                }
                finally {
                    this.commitLock.writeLock().unlock();
                    this.consumerConnector.shutdown();
                    this.commitThread.interrupt();
                    this.consumerExec.shutdownNow();
                }
            }
            catch (InterruptedException interruptedException) {
                Thread.currentThread().interrupt();
                Throwables.propagate((Throwable)interruptedException);
            }
            log.info("Finished clean shutdown.", new Object[0]);
        }
    }

    public void join() throws InterruptedException {
        this.commitThread.join();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void commit() throws InterruptedException {
        this.commitLock.writeLock().lockInterruptibly();
        try {
            long l = System.currentTimeMillis();
            Map<String, MessageCounters> map = this.writerController.flushAll();
            long l2 = System.currentTimeMillis();
            this.consumerConnector.commitOffsets();
            long l3 = System.currentTimeMillis();
            HashMap<String, MessageCounters> hashMap = new HashMap<String, MessageCounters>();
            for (Map.Entry<String, MessageCounters> entry : map.entrySet()) {
                hashMap.put(entry.getKey(), entry.getValue().difference(this.previousMessageCounters.get(entry.getKey())));
            }
            this.previousMessageCounters = map;
            log.info("Flushed %s pending messages in %sms and committed offsets in %sms.", new Object[]{hashMap.isEmpty() ? "0" : hashMap, l2 - l, l3 - l2});
        }
        finally {
            this.commitLock.writeLock().unlock();
        }
    }

    private Runnable createCommitRunnable() {
        return new Runnable(){

            @Override
            public void run() {
                long l = System.currentTimeMillis();
                try {
                    while (!Thread.currentThread().isInterrupted()) {
                        Thread.sleep(Math.max((long)KafkaConsumer.this.commitMillis - (System.currentTimeMillis() - l), 0L));
                        KafkaConsumer.this.commit();
                        l = System.currentTimeMillis();
                    }
                }
                catch (InterruptedException interruptedException) {
                    Thread.currentThread().interrupt();
                    log.info("Commit thread interrupted.", new Object[0]);
                }
                catch (Throwable throwable) {
                    log.error(throwable, "Commit thread failed!", new Object[0]);
                    throw Throwables.propagate((Throwable)throwable);
                }
                finally {
                    KafkaConsumer.this.stop();
                }
            }
        };
    }

    private void startConsumers() {
        List list = this.consumerConnector.createMessageStreamsByFilter(this.topicFilter, this.numThreads);
        for (final KafkaStream kafkaStream : list) {
            this.consumerExec.submit(new Runnable(){

                @Override
                public void run() {
                    try {
                        ConsumerIterator consumerIterator = kafkaStream.iterator();
                        while (consumerIterator.hasNext()) {
                            if (Thread.currentThread().isInterrupted()) {
                                throw new InterruptedException();
                            }
                            KafkaConsumer.this.commitLock.readLock().lockInterruptibly();
                            try {
                                MessageAndMetadata messageAndMetadata = (MessageAndMetadata)consumerIterator.next();
                                KafkaConsumer.this.writerController.getWriter(messageAndMetadata.topic()).send((byte[])messageAndMetadata.message());
                            }
                            finally {
                                KafkaConsumer.this.commitLock.readLock().unlock();
                            }
                        }
                    }
                    catch (InterruptedException interruptedException) {
                        log.info("Consumer thread interrupted.", new Object[0]);
                    }
                    catch (Throwable throwable) {
                        log.error(throwable, "Exception: ", new Object[0]);
                        throw Throwables.propagate((Throwable)throwable);
                    }
                    finally {
                        KafkaConsumer.this.stop();
                    }
                }
            });
        }
    }

    private static ConsumerConnector getConsumerConnector(Properties properties) {
        properties.setProperty("auto.commit.enable", "false");
        ConsumerConfig consumerConfig = new ConsumerConfig(properties);
        Preconditions.checkState((!consumerConfig.autoCommitEnable() ? 1 : 0) != 0, (Object)"autocommit must be off");
        return Consumer.createJavaConsumerConnector((ConsumerConfig)consumerConfig);
    }

    private static String buildTopicFilter(Map<String, DataSourceConfig<PropertiesBasedKafkaConfig>> map) {
        StringBuilder stringBuilder = new StringBuilder();
        for (Map.Entry<String, DataSourceConfig<PropertiesBasedKafkaConfig>> entry : map.entrySet()) {
            stringBuilder.append(String.format("(%s)|", ((PropertiesBasedKafkaConfig)entry.getValue().propertiesBasedConfig()).getTopicPattern()));
        }
        return stringBuilder.length() > 0 ? stringBuilder.substring(0, stringBuilder.length() - 1) : "";
    }
}

