/*
 * Decompiled with CFR 0.152.
 */
package io.druid.server.coordinator;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.metamx.emitter.EmittingLogger;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public class ReplicationThrottler {
    private static final EmittingLogger log = new EmittingLogger(ReplicationThrottler.class);
    private final int maxReplicants;
    private final int maxLifetime;
    private final Map<String, Boolean> replicatingLookup = Maps.newHashMap();
    private final Map<String, Boolean> terminatingLookup = Maps.newHashMap();
    private final ReplicatorSegmentHolder currentlyReplicating = new ReplicatorSegmentHolder();
    private final ReplicatorSegmentHolder currentlyTerminating = new ReplicatorSegmentHolder();

    public ReplicationThrottler(int maxReplicants, int maxLifetime) {
        this.maxReplicants = maxReplicants;
        this.maxLifetime = maxLifetime;
    }

    public void updateReplicationState(String tier) {
        this.update(tier, this.currentlyReplicating, this.replicatingLookup, "create");
    }

    public void updateTerminationState(String tier) {
        this.update(tier, this.currentlyTerminating, this.terminatingLookup, "terminate");
    }

    private void update(String tier, ReplicatorSegmentHolder holder, Map<String, Boolean> lookup, String type) {
        int size = holder.getNumProcessing(tier);
        if (size != 0) {
            log.info("[%s]: Replicant %s queue still has %d segments. Lifetime[%d]. Segments %s", new Object[]{tier, type, size, holder.getLifetime(tier), holder.getCurrentlyProcessingSegmentsAndHosts(tier)});
            holder.reduceLifetime(tier);
            lookup.put(tier, false);
            if (holder.getLifetime(tier) < 0) {
                log.makeAlert("[%s]: Replicant %s queue stuck after %d+ runs!", new Object[]{tier, type, this.maxLifetime}).addData("segments", holder.getCurrentlyProcessingSegmentsAndHosts(tier)).emit();
            }
        } else {
            log.info("[%s]: Replicant %s queue is empty.", new Object[]{tier, type});
            lookup.put(tier, true);
            holder.resetLifetime(tier);
        }
    }

    public boolean canCreateReplicant(String tier) {
        return this.replicatingLookup.get(tier) != false && !this.currentlyReplicating.isAtMaxReplicants(tier);
    }

    public boolean canDestroyReplicant(String tier) {
        return this.terminatingLookup.get(tier) != false && !this.currentlyTerminating.isAtMaxReplicants(tier);
    }

    public void registerReplicantCreation(String tier, String segmentId, String serverId) {
        this.currentlyReplicating.addSegment(tier, segmentId, serverId);
    }

    public void unregisterReplicantCreation(String tier, String segmentId, String serverId) {
        this.currentlyReplicating.removeSegment(tier, segmentId, serverId);
    }

    public void registerReplicantTermination(String tier, String segmentId, String serverId) {
        this.currentlyTerminating.addSegment(tier, segmentId, serverId);
    }

    public void unregisterReplicantTermination(String tier, String segmentId, String serverId) {
        this.currentlyTerminating.removeSegment(tier, segmentId, serverId);
    }

    private class ReplicatorSegmentHolder {
        private final Map<String, ConcurrentHashMap<String, String>> currentlyProcessingSegments = Maps.newHashMap();
        private final Map<String, Integer> lifetimes = Maps.newHashMap();

        private ReplicatorSegmentHolder() {
        }

        public boolean isAtMaxReplicants(String tier) {
            ConcurrentHashMap<String, String> segments = this.currentlyProcessingSegments.get(tier);
            return segments != null && segments.size() >= ReplicationThrottler.this.maxReplicants;
        }

        public void addSegment(String tier, String segmentId, String serverId) {
            ConcurrentHashMap<String, String> segments = this.currentlyProcessingSegments.get(tier);
            if (segments == null) {
                segments = new ConcurrentHashMap();
                this.currentlyProcessingSegments.put(tier, segments);
            }
            if (!this.isAtMaxReplicants(tier)) {
                segments.put(segmentId, serverId);
            }
        }

        public void removeSegment(String tier, String segmentId, String serverId) {
            Map segments = this.currentlyProcessingSegments.get(tier);
            if (segments != null) {
                segments.remove(segmentId);
            }
        }

        public int getNumProcessing(String tier) {
            Map segments = this.currentlyProcessingSegments.get(tier);
            return segments == null ? 0 : segments.size();
        }

        public int getLifetime(String tier) {
            Integer lifetime = this.lifetimes.get(tier);
            if (lifetime == null) {
                lifetime = ReplicationThrottler.this.maxLifetime;
                this.lifetimes.put(tier, lifetime);
            }
            return lifetime;
        }

        public void reduceLifetime(String tier) {
            Integer lifetime = this.lifetimes.get(tier);
            if (lifetime == null) {
                lifetime = ReplicationThrottler.this.maxLifetime;
                this.lifetimes.put(tier, lifetime);
            }
            lifetime = lifetime - 1;
            this.lifetimes.put(tier, lifetime);
        }

        public void resetLifetime(String tier) {
            this.lifetimes.put(tier, ReplicationThrottler.this.maxLifetime);
        }

        public List<String> getCurrentlyProcessingSegmentsAndHosts(String tier) {
            Map segments = this.currentlyProcessingSegments.get(tier);
            ArrayList retVal = Lists.newArrayList();
            for (Map.Entry entry : segments.entrySet()) {
                retVal.add(String.format("%s ON %s", entry.getKey(), entry.getValue()));
            }
            return retVal;
        }
    }
}

