/*
 * Decompiled with CFR 0.152.
 */
package io.druid.segment.realtime;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.Maps;
import com.google.common.io.Closeables;
import com.google.inject.Inject;
import com.metamx.common.exception.FormattedException;
import com.metamx.common.lifecycle.LifecycleStart;
import com.metamx.common.lifecycle.LifecycleStop;
import com.metamx.emitter.EmittingLogger;
import io.druid.data.input.Firehose;
import io.druid.data.input.InputRow;
import io.druid.query.FinalizeResultsQueryRunner;
import io.druid.query.Query;
import io.druid.query.QueryRunner;
import io.druid.query.QueryRunnerFactory;
import io.druid.query.QueryRunnerFactoryConglomerate;
import io.druid.query.QuerySegmentWalker;
import io.druid.query.QueryToolChest;
import io.druid.query.SegmentDescriptor;
import io.druid.segment.realtime.FireDepartment;
import io.druid.segment.realtime.FireDepartmentConfig;
import io.druid.segment.realtime.FireDepartmentMetrics;
import io.druid.segment.realtime.Schema;
import io.druid.segment.realtime.plumber.Plumber;
import io.druid.segment.realtime.plumber.Sink;
import java.io.Closeable;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import org.joda.time.DateTime;
import org.joda.time.Interval;
import org.joda.time.Period;
import org.joda.time.ReadablePeriod;

public class RealtimeManager
implements QuerySegmentWalker {
    private static final EmittingLogger log = new EmittingLogger(RealtimeManager.class);
    private final List<FireDepartment> fireDepartments;
    private final QueryRunnerFactoryConglomerate conglomerate;
    private final Map<String, FireChief> chiefs;

    @Inject
    public RealtimeManager(List<FireDepartment> fireDepartments, QueryRunnerFactoryConglomerate conglomerate) {
        this.fireDepartments = fireDepartments;
        this.conglomerate = conglomerate;
        this.chiefs = Maps.newHashMap();
    }

    @LifecycleStart
    public void start() throws IOException {
        for (FireDepartment fireDepartment : this.fireDepartments) {
            Schema schema = fireDepartment.getSchema();
            FireChief chief = new FireChief(fireDepartment);
            this.chiefs.put(schema.getDataSource(), chief);
            chief.setName(String.format("chief-%s", schema.getDataSource()));
            chief.setDaemon(true);
            chief.init();
            chief.start();
        }
    }

    @LifecycleStop
    public void stop() {
        for (FireChief chief : this.chiefs.values()) {
            Closeables.closeQuietly((Closeable)chief);
        }
    }

    public FireDepartmentMetrics getMetrics(String datasource) {
        FireChief chief = this.chiefs.get(datasource);
        if (chief == null) {
            return null;
        }
        return chief.getMetrics();
    }

    public <T> QueryRunner<T> getQueryRunnerForIntervals(Query<T> query, Iterable<Interval> intervals) {
        FireChief chief = this.chiefs.get(query.getDataSource());
        return chief == null ? new QueryRunner<T>() : chief.getQueryRunner(query);
    }

    public <T> QueryRunner<T> getQueryRunnerForSegments(Query<T> query, Iterable<SegmentDescriptor> specs) {
        FireChief chief = this.chiefs.get(query.getDataSource());
        return chief == null ? new QueryRunner<T>() : chief.getQueryRunner(query);
    }

    private class FireChief
    extends Thread
    implements Closeable {
        private final FireDepartment fireDepartment;
        private final FireDepartmentMetrics metrics;
        private volatile FireDepartmentConfig config = null;
        private volatile Firehose firehose = null;
        private volatile Plumber plumber = null;
        private volatile boolean normalExit = true;

        public FireChief(FireDepartment fireDepartment) {
            this.fireDepartment = fireDepartment;
            this.metrics = fireDepartment.getMetrics();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void init() throws IOException {
            this.config = this.fireDepartment.getConfig();
            FireChief fireChief = this;
            synchronized (fireChief) {
                try {
                    log.info("Calling the FireDepartment and getting a Firehose.", new Object[0]);
                    this.firehose = this.fireDepartment.connect();
                    log.info("Firehose acquired!", new Object[0]);
                    log.info("Someone get us a plumber!", new Object[0]);
                    this.plumber = this.fireDepartment.findPlumber();
                    log.info("We have our plumber!", new Object[0]);
                }
                catch (IOException e) {
                    throw Throwables.propagate((Throwable)e);
                }
            }
        }

        public FireDepartmentMetrics getMetrics() {
            return this.metrics;
        }

        @Override
        public void run() {
            this.verifyState();
            Period intermediatePersistPeriod = this.config.getIntermediatePersistPeriod();
            try {
                this.plumber.startJob();
                long nextFlush = new DateTime().plus((ReadablePeriod)intermediatePersistPeriod).getMillis();
                while (this.firehose.hasMore()) {
                    try {
                        InputRow inputRow;
                        try {
                            inputRow = this.firehose.nextRow();
                        }
                        catch (Exception e) {
                            log.info((Throwable)e, "thrown away line due to exception", new Object[0]);
                            this.metrics.incrementThrownAway();
                            continue;
                        }
                        Sink sink = this.plumber.getSink(inputRow.getTimestampFromEpoch());
                        if (sink == null) {
                            this.metrics.incrementThrownAway();
                            log.debug("Throwing away event[%s]", new Object[]{inputRow});
                            if (System.currentTimeMillis() <= nextFlush) continue;
                            this.plumber.persist(this.firehose.commit());
                            nextFlush = new DateTime().plus((ReadablePeriod)intermediatePersistPeriod).getMillis();
                            continue;
                        }
                        int currCount = sink.add(inputRow);
                        if (currCount >= this.config.getMaxRowsInMemory() || System.currentTimeMillis() > nextFlush) {
                            this.plumber.persist(this.firehose.commit());
                            nextFlush = new DateTime().plus((ReadablePeriod)intermediatePersistPeriod).getMillis();
                        }
                        this.metrics.incrementProcessed();
                    }
                    catch (FormattedException e) {
                        log.info((Throwable)e, "unparseable line: %s", new Object[]{e.getDetails()});
                        this.metrics.incrementUnparseable();
                    }
                }
            }
            catch (RuntimeException e) {
                log.makeAlert((Throwable)e, "RuntimeException aborted realtime processing[%s]", new Object[]{this.fireDepartment.getSchema().getDataSource()}).emit();
                this.normalExit = false;
                throw e;
            }
            catch (Error e) {
                log.makeAlert((Throwable)e, "Exception aborted realtime processing[%s]", new Object[]{this.fireDepartment.getSchema().getDataSource()}).emit();
                this.normalExit = false;
                throw e;
            }
            finally {
                Closeables.closeQuietly((Closeable)this.firehose);
                if (this.normalExit) {
                    this.plumber.finishJob();
                    this.plumber = null;
                    this.firehose = null;
                }
            }
        }

        private void verifyState() {
            Preconditions.checkNotNull((Object)this.config, (Object)"config is null, init() must be called first.");
            Preconditions.checkNotNull((Object)this.firehose, (Object)"firehose is null, init() must be called first.");
            Preconditions.checkNotNull((Object)this.plumber, (Object)"plumber is null, init() must be called first.");
            log.info("FireChief[%s] state ok.", new Object[]{this.fireDepartment.getSchema().getDataSource()});
        }

        public <T> QueryRunner<T> getQueryRunner(Query<T> query) {
            QueryRunnerFactory factory = RealtimeManager.this.conglomerate.findFactory(query);
            QueryToolChest toolChest = factory.getToolchest();
            return new FinalizeResultsQueryRunner(this.plumber.getQueryRunner(query), toolChest);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void close() throws IOException {
            FireChief fireChief = this;
            synchronized (fireChief) {
                if (this.firehose != null) {
                    this.normalExit = false;
                    this.firehose.close();
                }
            }
        }
    }
}

