/*
 * Decompiled with CFR 0.152.
 */
package io.druid.client.cache;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import com.metamx.common.logger.Logger;
import io.druid.client.cache.Cache;
import io.druid.client.cache.CacheStats;
import io.druid.client.cache.LZ4Transcoder;
import io.druid.client.cache.MemcachedCacheConfig;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.Nullable;
import net.spy.memcached.AddrUtil;
import net.spy.memcached.ConnectionFactoryBuilder;
import net.spy.memcached.DefaultHashAlgorithm;
import net.spy.memcached.FailureMode;
import net.spy.memcached.HashAlgorithm;
import net.spy.memcached.MemcachedClient;
import net.spy.memcached.MemcachedClientIF;
import net.spy.memcached.internal.BulkFuture;
import net.spy.memcached.transcoders.Transcoder;
import org.apache.commons.codec.digest.DigestUtils;

public class MemcachedCache
implements Cache {
    private static final Logger log = new Logger(MemcachedCache.class);
    private final int timeout;
    private final int expiration;
    private final String memcachedPrefix;
    private final MemcachedClientIF client;
    private final AtomicLong hitCount = new AtomicLong(0L);
    private final AtomicLong missCount = new AtomicLong(0L);
    private final AtomicLong timeoutCount = new AtomicLong(0L);
    private final AtomicLong errorCount = new AtomicLong(0L);
    public static final int MAX_PREFIX_LENGTH = 168;

    public static MemcachedCache create(MemcachedCacheConfig config) {
        try {
            LZ4Transcoder transcoder = new LZ4Transcoder(config.getMaxObjectSize());
            transcoder.setCompressionThreshold(0);
            return new MemcachedCache((MemcachedClientIF)new MemcachedClient(new ConnectionFactoryBuilder().setProtocol(ConnectionFactoryBuilder.Protocol.BINARY).setHashAlg((HashAlgorithm)DefaultHashAlgorithm.FNV1A_64_HASH).setLocatorType(ConnectionFactoryBuilder.Locator.CONSISTENT).setDaemon(true).setFailureMode(FailureMode.Cancel).setTranscoder((Transcoder)transcoder).setShouldOptimize(true).setOpQueueMaxBlockTime((long)config.getTimeout()).setOpTimeout((long)config.getTimeout()).build(), AddrUtil.getAddresses((String)config.getHosts())), config.getMemcachedPrefix(), config.getTimeout(), config.getExpiration());
        }
        catch (IOException e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    MemcachedCache(MemcachedClientIF client, String memcachedPrefix, int timeout, int expiration) {
        Preconditions.checkArgument((memcachedPrefix.length() <= 168 ? 1 : 0) != 0, (String)"memcachedPrefix length [%d] exceeds maximum length [%d]", (Object[])new Object[]{memcachedPrefix.length(), 168});
        this.timeout = timeout;
        this.expiration = expiration;
        this.client = client;
        this.memcachedPrefix = memcachedPrefix;
    }

    @Override
    public CacheStats getStats() {
        return new CacheStats(this.hitCount.get(), this.missCount.get(), 0L, 0L, 0L, this.timeoutCount.get(), this.errorCount.get());
    }

    @Override
    public byte[] get(Cache.NamedKey key) {
        Future future;
        try {
            future = this.client.asyncGet(MemcachedCache.computeKeyHash(this.memcachedPrefix, key));
        }
        catch (IllegalStateException e) {
            this.errorCount.incrementAndGet();
            log.warn((Throwable)e, "Unable to queue cache operation", new Object[0]);
            return null;
        }
        try {
            byte[] bytes = (byte[])future.get(this.timeout, TimeUnit.MILLISECONDS);
            if (bytes != null) {
                this.hitCount.incrementAndGet();
            } else {
                this.missCount.incrementAndGet();
            }
            return bytes == null ? null : MemcachedCache.deserializeValue(key, bytes);
        }
        catch (TimeoutException e) {
            this.timeoutCount.incrementAndGet();
            future.cancel(false);
            return null;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw Throwables.propagate((Throwable)e);
        }
        catch (ExecutionException e) {
            this.errorCount.incrementAndGet();
            log.warn((Throwable)e, "Exception pulling item from cache", new Object[0]);
            return null;
        }
    }

    @Override
    public void put(Cache.NamedKey key, byte[] value) {
        try {
            this.client.set(MemcachedCache.computeKeyHash(this.memcachedPrefix, key), this.expiration, (Object)MemcachedCache.serializeValue(key, value));
        }
        catch (IllegalStateException e) {
            this.errorCount.incrementAndGet();
            log.warn((Throwable)e, "Unable to queue cache operation", new Object[0]);
        }
    }

    private static byte[] serializeValue(Cache.NamedKey key, byte[] value) {
        byte[] keyBytes = key.toByteArray();
        return ByteBuffer.allocate(4 + keyBytes.length + value.length).putInt(keyBytes.length).put(keyBytes).put(value).array();
    }

    private static byte[] deserializeValue(Cache.NamedKey key, byte[] bytes) {
        ByteBuffer buf = ByteBuffer.wrap(bytes);
        int keyLength = buf.getInt();
        byte[] keyBytes = new byte[keyLength];
        buf.get(keyBytes);
        byte[] value = new byte[buf.remaining()];
        buf.get(value);
        Preconditions.checkState((boolean)Arrays.equals(keyBytes, key.toByteArray()), (Object)"Keys do not match, possible hash collision?");
        return value;
    }

    @Override
    public Map<Cache.NamedKey, byte[]> getBulk(Iterable<Cache.NamedKey> keys) {
        BulkFuture future;
        ImmutableMap keyLookup = Maps.uniqueIndex(keys, (Function)new Function<Cache.NamedKey, String>(){

            public String apply(@Nullable Cache.NamedKey input) {
                return MemcachedCache.computeKeyHash(MemcachedCache.this.memcachedPrefix, input);
            }
        });
        HashMap results = Maps.newHashMap();
        try {
            future = this.client.asyncGetBulk(keyLookup.keySet());
        }
        catch (IllegalStateException e) {
            this.errorCount.incrementAndGet();
            log.warn((Throwable)e, "Unable to queue cache operation", new Object[0]);
            return results;
        }
        try {
            Map some = (Map)future.getSome((long)this.timeout, TimeUnit.MILLISECONDS);
            if (future.isTimeout()) {
                future.cancel(false);
                this.timeoutCount.incrementAndGet();
            }
            this.missCount.addAndGet(keyLookup.size() - some.size());
            this.hitCount.addAndGet(some.size());
            for (Map.Entry entry : some.entrySet()) {
                Cache.NamedKey key = (Cache.NamedKey)keyLookup.get(entry.getKey());
                byte[] value = (byte[])entry.getValue();
                results.put(key, value == null ? null : MemcachedCache.deserializeValue(key, value));
            }
            return results;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw Throwables.propagate((Throwable)e);
        }
        catch (ExecutionException e) {
            this.errorCount.incrementAndGet();
            log.warn((Throwable)e, "Exception pulling item from cache", new Object[0]);
            return results;
        }
    }

    @Override
    public void close(String namespace) {
    }

    private static String computeKeyHash(String memcachedPrefix, Cache.NamedKey key) {
        return memcachedPrefix + ":" + DigestUtils.sha1Hex((String)key.namespace) + ":" + DigestUtils.sha1Hex((byte[])key.key);
    }
}

